//**********************************************************************************************
//* WebMail Library
//* ---------------
//* A collection of internet e-mail protocol (POP, SMTP, IMAP) 
//* encapsulation classes for Ultimate++
//* 

#include "WebMail.h"

String POP3Mail::SendRecvOK(String s, bool multiline, int timeout)
{
	Send(s, timeout);
	String out = Recv(s, multiline, timeout);
	if(out.StartsWith("-ERR"))
		throw(out);
	else
	if(out.StartsWith("+OK")) {
		const char *p = out.Begin() + 3;
		out = p;
	}
	return out;
}

void POP3Mail::Connect(String host, int port, String user, String pass, int ssltype, bool apop)
{
	// Connect.
	MailSocket::UseSSL(ssltype);
	MailSocket::Connect(host, port, user, pass);
	String hello = SendRecvOK(Null);
	// Authorize.
	state = AUTHORIZATION;
	// Check if APOP supported.
	if(apop) {
		String hash;
		state = false;
		for(const char* p = hello.Begin(); p < hello.Last(); p++) {
			if(*p == '<' || state) {
				hash.Cat(p);
				if(*p == '>')
					break;
				state = true;
			}
		}
		if(IsNull(hash))
			throw Exc(t_("Server does not support APOP authentication."));
		else  {
			hash << pass;
			SendRecvOK("APOP " + user + " " + MD5Digest(hash) + "\r\n");
		}
	}
	else {
	SendRecvOK("USER " + user + "\r\n");
	SendRecvOK("PASS " + pass + "\r\n");
	}
	// successfully logged in. We can change state.
	state = TRANSACTION;	
}

void POP3Mail::Disconnect()
{
	Quit();
	state = DISCONNECTED;
}


EMail POP3Mail::GetMail(int index)
{
	return EMail(SendRecvOK(Format("RETR %d\r\n", index), true));
}

String POP3Mail::GetMailHeader(int index, int length)
{
	return SendRecvOK(Format("TOP %d %d\r\n", index, length), true);
}

String POP3Mail::GetMailGuid(int index)
{
	char id[64], cnt[8];
	sscanf(SendRecvOK(Format("UIDL %d\r\n", index)),
		 	"%s %s", 
		 	cnt, id
		 	);
	return id;
}

void POP3Mail::GetMailStats(int& count, double& totalsize)
{
	char cnt[8], tsz[32];
	sscanf(SendRecvOK("STAT \r\n"), 
			"%s %s", 
			cnt, tsz
			);
   	count = StrInt(cnt);
  	totalsize = StrDbl(tsz);
}

double POP3Mail::GetMailSize(int index)
{
	char cnt[8], sz[16];
	sscanf(SendRecvOK(Format("LIST %d\r\n", index)),
			"%s %s", 
			cnt, sz
			);
	return StrDbl(sz);
}

void POP3Mail::GetMailList(VectorMap<int, double>& list)
{
	String s = SendRecvOK("LIST \r\n", true), out;
	char cnt[8], sz[32];
	int i = 0;
	bool val = true;
	const char *p = s;
	
	if(*p == ' ') {
		p++;
		sscanf(p, "%s %s", cnt, sz);
		p = s.Begin() + String(p).Find("\r\n") + 1;
	}
	for(p; p < s.Last(); p++) {
			if(*p == '\r' && *(p + 1) == '\n') {
				if(*(p + 2) == '.')
					break;
				else val = true;
			}
			if(val) {
				sscanf(p, "%s %s", cnt, sz);
				list.Add(StrInt(cnt), StrDbl(sz));
				val = false;
			
			}
	}
}

void POP3Mail::GetMailGuids(VectorMap<int, String>& guids)
{
	String uids = SendRecvOK("UIDL \r\n", true);
	char cnt[8], guid[64];
	bool val = true;
	for(const char *p = uids.Begin(); p < uids.Last(); p++) {
		if(*p == '\r' && *(p + 1) == '\n') { 
			if(*(p + 2) == '.') 
				break;
			else val = true;;
		}
		if(val) {	
			sscanf(p, "%s %s", cnt, guid);
				guids.Add(StrInt(cnt), String(guid));
			val = false;
		}
	}
}

bool POP3Mail::ReadInbox(Vector<EMail>& mailbox, int length)
{
	try {
		// Connect and login to POP3 server.
		Connect(host0, port0, user0, pass0, sockettype, useapop);
		// Check the POP3 state:
		if(IsTransacting()) {
			
			// Get inbox statistics and see if there are any mails.
			int count = 0; double total = 0;
			GetMailStats(count, total);
			if(count > 0) {
				// Retreive unique-ids for identification.
				VectorMap<int, String> guids;
				VectorMap<int, double> list;
				GetMailList(list);
				GetMailGuids(guids);
				if(guids.GetCount() > 0 && guids.GetCount() == count) {
					// Retreive mails from server. 
					// length <  0: retreive full message.
					// length =  0: retreive only headers.
					// lentth >	 0: retreive (length) lines of mails.
					for(int i = 0; i < count; i++) {
						EMail mail((length < 0 ? GetMail(i + 1) : GetMailHeader(i + 1, length)));
						mail.Guid()	= guids[i];
						mailbox.Add(mail); 
					}
				}
			}
			// Logout and disconnect from POP3 server.
		}
		Disconnect();
	}
	catch (Exc e) {
		error = e;
		return false;
	}
	return true;
}

void POP3Mail::DeleteMail(int index)
{
	SendRecvOK(Format("DELE %d\r\n", index));
}

void POP3Mail::Undo()
{
	SendRecvOK("RSET\r\n");	
}

void POP3Mail::Noop()
{
	SendRecvOK("NOOP\r\n");	
}

void POP3Mail::Quit()
{
	SendRecvOK("QUIT\r\n");
	if(IsAuthorizing() || IsUpdating())	
		state = DISCONNECTED;
	else if(IsTransacting())
		state = UPDATE;
}

POP3Mail::POP3Mail() :
useapop(false)
{
	//default port.
	Port(110); 
	// default state.
	state = DISCONNECTED;
	// default socket layer type.
	sockettype = MailSocket::NOSSL;
}